<?php
 
namespace App\Http\Controllers\Auth;
 
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\LoginRequest;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    protected $maxAttempts = 2;   // ログイン認証試行許可回数、＋1回失敗でロック
    protected $decayMinutes = 1;  // ロック時間（分）

    /**
     * 認証の試行を処理
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function authenticate(LoginRequest $request)
    {
        // ログイン認証試行許可回数チェック(AuthenticatesUsers)
        if (method_exists($this, 'hasTooManyLoginAttempts') &&
            $this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);
            
            // ログイン認証試行オーバーのためロックする
            return $this->sendLockoutResponse($request);
        }
        
        // 資格情報を取得
        $credentials = $request->validated();

        // ログイン認証
        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();

            // 認証成功
            return redirect()->route('home');
        }

        // ログイン認証試行インクリメント
        $this->incrementLoginAttempts($request);

        // 認証失敗
        return $this->sendFailedLoginResponse($request);
    }

    // sendFailedLoginResponse Override
    protected function sendFailedLoginResponse(Request $request)
    {
        throw ValidationException::withMessages([
            $this->username() => [\Lang::get('system.login.failed')],
        ]);
    }

    // throttleKey Override
    protected function throttleKey(Request $request)
    {
        return $request->ip();
    }

    /**
     * ユーザーをアプリケーションからログアウトさせる
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        Auth::logout();
    
        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    // Username Override
    public function username()
    {
        return 'account';
    }

}
