<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\BillingRequest;
use App\Models\Billing;
use App\Http\Controllers\Common\SendErrorMessage;

class BillingController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }

        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.billing.name'),
          \Lang::get('db.billing.address'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $billingQuery = Billing::query();
        if (!empty($params['search'])) {
          $billingQuery = $billingQuery->where('name', 'like', '%'.$params['search'].'%');
          $billingQuery = $billingQuery->orWhere('zip_code', '=', $params['search']);
          $billingQuery = $billingQuery->orWhere('address_1', 'like', '%'.$params['search'].'%');
          $billingQuery = $billingQuery->orWhere('address_2', 'like', '%'.$params['search'].'%');
        }

        // DBよりBillingテーブルの値を検索
        $billings = $billingQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「billing/index」に渡す
        return view('billing/index')
            ->with([
                'billings' => $billings,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$billingを渡す
        $billing = new Billing();
        return view('billing/create', compact('billing'));
    }
    
    public function store(BillingRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $billing = new Billing();
        $billing->name = $request->name;
        $billing->zip_code = $request->zip_code;
        $billing->address_1 = $request->address_1;
        $billing->address_2 = $request->address_2;
        $billing->phone_number = $request->phone_number;
        $billing->person_in_charge = $request->person_in_charge;
        $billing->note = $request->note;
        $billing->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つBillingの情報を取得
        $billing = Billing::findOrFail($id);

        // 取得した値をビュー「billing/edit」に渡す
        return view('billing/edit', compact('billing'));
    }

    public function update(BillingRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $billing = Billing::findOrFail($id);
        $billing->name = $request->name;
        $billing->zip_code = $request->zip_code;
        $billing->address_1 = $request->address_1;
        $billing->address_2 = $request->address_2;
        $billing->phone_number = $request->phone_number;
        $billing->person_in_charge = $request->person_in_charge;
        $billing->note = $request->note;
        $billing->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $billing = Billing::findOrFail($id);
        $billing->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

}
