<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\CarRequest;
use App\Models\Car;
use App\Http\Controllers\Common\SendErrorMessage;

class CarController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }
        
        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.car.number'),
          \Lang::get('db.car.maker'),
          \Lang::get('db.car.name'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $carQuery = Car::query();
        if (!empty($params['search'])) {
          $carQuery = $carQuery->where('number', 'like', '%'.$params['search'].'%');
          $carQuery = $carQuery->orWhere('maker', 'like', '%'.$params['search'].'%');
          $carQuery = $carQuery->orWhere('name', 'like', '%'.$params['search'].'%');
        }

        // DBよりCarテーブルの値を検索
        $cars = $carQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「car/index」に渡す
        return view('car/index')
            ->with([
                'cars' => $cars,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$carを渡す
        $car = new Car();
        return view('car/create', compact('car'));
    }
    
    public function store(CarRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $car = new Car();
        $car->name = $request->name;
        $car->format = $request->format;
        $car->size = $request->size;
        $car->number = $request->number;
        $car->capacity = $request->capacity;
        $car->use = $request->use;
        $car->maker = $request->maker;
        $car->inspection_at = $request->inspection_at;
        if (!empty($request->inspection_next_at)) {
          $car->inspection_next_at = $request->inspection_next_at.'-01';
        } else {
          $car->inspection_next_at = null;
        }
        $car->note = $request->note;
        $car->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つCarの情報を取得
        $car = Car::findOrFail($id);

        // 取得した値をビュー「car/edit」に渡す
        return view('car/edit', compact('car'));
    }

    public function update(CarRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $car = Car::findOrFail($id);
        $car->name = $request->name;
        $car->format = $request->format;
        $car->size = $request->size;
        $car->number = $request->number;
        $car->capacity = $request->capacity;
        $car->use = $request->use;
        $car->maker = $request->maker;
        $car->inspection_at = $request->inspection_at;
        if (!empty($request->inspection_next_at)) {
          $car->inspection_next_at = $request->inspection_next_at.'-01';
        } else {
          $car->inspection_next_at = null;
        }
        $car->note = $request->note;
        $car->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $car = Car::findOrFail($id);
        $car->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

}
