<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\ExampleRequest;
use App\Models\Example;
use App\Http\Controllers\Common\SendErrorMessage;

class ExampleController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }
        
        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.example.name'),
          \Lang::get('db.example.author'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $exampleQuery = Example::query();
        if (!empty($params['search'])) {
          $exampleQuery = $exampleQuery->where('name', 'like', '%'.$params['search'].'%');
          $exampleQuery = $exampleQuery->orWhere('author', 'like', '%'.$params['search'].'%');
        }

        // DBよりExampleテーブルの値を検索
        $examples = $exampleQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「example/index」に渡す
        return view('example/index')
            ->with([
                'examples' => $examples,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$exampleを渡す
        $example = new Example();
        return view('example/create', compact('example'));
    }
    
    public function store(ExampleRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $example = new Example();
        $example->name = $request->name;
        $example->price = $request->price;
        $example->author = $request->author;
        $example->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つExampleの情報を取得
        $example = Example::findOrFail($id);

        // 取得した値をビュー「example/edit」に渡す
        return view('example/edit', compact('example'));
    }

    public function update(ExampleRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $example = Example::findOrFail($id);
        $example->name = $request->name;
        $example->price = $request->price;
        $example->author = $request->author;
        $example->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $example = Example::findOrFail($id);
        $example->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

}
