<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\ItemRequest;
use App\Models\Item;
use App\Http\Controllers\Common\SendErrorMessage;

class ItemController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }
        
        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.item.name'),
          \Lang::get('db.item.note'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $itemQuery = Item::query();
        if (!empty($params['search'])) {
          $itemQuery = $itemQuery->where('name', 'like', '%'.$params['search'].'%');
          $itemQuery = $itemQuery->orWhere('note', 'like', '%'.$params['search'].'%');
        }

        // DBよりItemテーブルの値を検索
        $items = $itemQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「item/index」に渡す
        return view('item/index')
            ->with([
                'items' => $items,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$itemを渡す
        $item = new Item();
        return view('item/create', compact('item'));
    }
    
    public function store(ItemRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $item = new Item();
        $item->name = $request->name;
        $item->note = $request->note;
        $item->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つItemの情報を取得
        $item = Item::findOrFail($id);

        // 取得した値をビュー「item/edit」に渡す
        return view('item/edit', compact('item'));
    }

    public function update(ItemRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $item = Item::findOrFail($id);
        $item->name = $request->name;
        $item->note = $request->note;
        $item->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $item = Item::findOrFail($id);
        $item->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

}
