<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\PartnerRequest;
use App\Models\Partner;
use App\Http\Controllers\Common\SendErrorMessage;

class PartnerController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }

        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.partner.name'),
          \Lang::get('db.partner.company_name'),
          \Lang::get('db.partner.note'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $partnerQuery = Partner::query();
        if (!empty($params['search'])) {
          $partnerQuery = $partnerQuery->where('name', 'like', '%'.$params['search'].'%');
          $partnerQuery = $partnerQuery->orWhere('company_name', 'like', '%'.$params['search'].'%');
          $partnerQuery = $partnerQuery->orWhere('note', 'like', '%'.$params['search'].'%');
        }

        // DBよりPartnerテーブルの値を検索
        $partners = $partnerQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「partner/index」に渡す
        return view('partner/index')
            ->with([
                'partners' => $partners,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$partnerを渡す
        $partner = new Partner();
        return view('partner/create', compact('partner'));
    }
    
    public function store(PartnerRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $partner = new Partner();
        $partner->name = $request->name;
        $partner->company_name = $request->company_name;
        $partner->quantity = $request->quantity;
        $partner->note = $request->note;
        $partner->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つPartnerの情報を取得
        $partner = Partner::findOrFail($id);

        // 取得した値をビュー「partner/edit」に渡す
        return view('partner/edit', compact('partner'));
    }

    public function update(PartnerRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $partner = Partner::findOrFail($id);
        $partner->name = $request->name;
        $partner->company_name = $request->company_name;
        $partner->quantity = $request->quantity;
        $partner->note = $request->note;
        $partner->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $partner = Partner::findOrFail($id);
        $partner->delete();

        return redirect(session()->get('checkpoint') ?? './');
    }

}
