<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Billing;
use App\Models\ReportWaypoint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Common\SendErrorMessage;

class PdfBillingController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['from_at'])) {
          $params['from_at'] = date('Y-m-d', strtotime('first day of this month'));
        }
        if (empty($params['to_at'])) {
          $params['to_at'] = date('Y-m-d', strtotime('last day of this month'));
        }

        // ページネーション設定
        // $page_max = config('const.DEFAULT_PAGE_MAX');
        $page_max = 10;

        $report_waypointQuery = ReportWaypoint::query();
        $report_waypointQuery = $report_waypointQuery->whereBetween('reports.start_at', [$params['from_at'].' 00:00:00', $params['to_at'].' 23:59:59']);

        $report_waypoints = $report_waypointQuery
            ->leftJoin('reports', 'report_waypoints.report_id','=','reports.id')
            ->select('report_waypoints.billing_id', 
              DB::raw("COUNT(report_waypoints.billing_id) as count"),
              DB::raw("SUM(ROUND(report_waypoints.quantity * report_waypoints.unit_price)) as sum_total_price")
            )
            ->groupBy('report_waypoints.billing_id')
            ->orderBy('sum_total_price', 'desc')
            ->paginate($page_max);

        // 取得した値をビュー「csv/billing/index」に渡す
        return view('pdf/billing/index')
            ->with([
                'params' => $params,
                'report_waypoints' => $report_waypoints,
            ]);
    }

    public function details(Request $request, $id)
    {
        // チェックポイント保存
        session(['checkpoint-second' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['default_tax'])) {
          $params['default_tax'] = 10;  // 税率の初期値（税率変わった場合はココを変更、過渡期は直接数値を変更してもらえばOK）
        }

        $report_waypointQuery = ReportWaypoint::query();
        $report_waypointQuery = $report_waypointQuery->where('report_waypoints.billing_id', '=', $id);
        $report_waypointQuery = $report_waypointQuery->whereBetween('reports.start_at', [$params['from_at'].' 00:00:00', $params['to_at'].' 23:59:59']);

        $report_waypoints = $report_waypointQuery
            ->leftJoin('reports', 'report_waypoints.report_id','=','reports.id')
            ->select('report_waypoints.*', DB::raw('DATE_FORMAT(reports.start_at, "%Y/%m/%d") AS date'))
            ->orderBy('date', 'asc')
            ->orderBy('report_waypoints.id', 'asc')
            ->get();

        // 取得した値をビュー「csv/billing/details」に渡す
        return view('pdf/billing/details')
            ->with([
                'params' => $params,
                'report_waypoints' => $report_waypoints,
            ]);
    }

    public function output(Request $request)
    {
      // バリデーション
      $validated = $request->validate([
        'last_month_invoice' => 'nullable|digits_between:1,7',
        'last_month_payment' => 'nullable|digits_between:1,7',
        'this_month_carried' => 'nullable|digits_between:1,7',
        'this_month_adjust' => 'nullable|digits_between:1,7',
        'tax' => 'required|integer|min:1|max:99',
      ], [], [
        'last_month_invoice' => '前月御請求額',
        'last_month_payment' => '御入金額',
        'this_month_carried' => '繰越金額',
        'this_month_adjust' => '調整額',
        'tax' => '消費税率',
      ]);

      // パラメータ取得
      $params = $request->all();

      // データ取得
      $invoices = DB::table('report_waypoints')
      ->selectRaw('billings.id AS billing_id')
      ->selectRaw('billings.name AS billing_name')
      ->selectRaw('billings.zip_code AS billing_zip_code')
      ->selectRaw('billings.address_1 AS billing_address_1')
      ->selectRaw('billings.address_2 AS billing_address_2')
      ->selectRaw('DATE_FORMAT(reports.start_at, "%Y/%m/%d") AS date')
      ->selectRaw('report_waypoints.take_up AS item_1')
      ->selectRaw('report_waypoints.take_down AS item_2')
      ->selectRaw('report_waypoints.quantity AS quantity')
      ->selectRaw('report_waypoints.unit AS unit')
      ->selectRaw('report_waypoints.unit_price AS unit_price')
      ->leftJoin('reports', 'report_waypoints.report_id','=','reports.id')
      ->leftJoin('billings', 'report_waypoints.billing_id','=','billings.id')
      ->leftJoin('cars', 'reports.car_id','=','cars.id')
      ->leftJoin('types', 'report_waypoints.type_id','=','types.id')
      ->whereBetween('reports.start_at', [$params['from_at'].' 00:00:00', $params['to_at'].' 23:59:59'])
      ->where('billing_id', '=', ($params['billing_id']))
      ->orderBy('report_waypoints.billing_id', 'asc')
      ->orderBy('date', 'asc')
      ->orderBy('report_waypoints.id', 'asc');

      // 請求先データ取得
      $billing = Billing::findOrFail($params['billing_id']);

      if ($invoices->count() === 0) {
        // 対象データなし
        // エラーメッセージ
        $error_messages = [];
        if (empty($request->report_id)) {
          $error_messages[] = $billing->name.' の '.date('Y年n月j日', strtotime($params['from_at'])).' ～ '.date('Y年n月j日', strtotime($params['to_at'])).' のデータはありません。';
        }
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }
      }

      $total_price = 0;
      foreach ($invoices->cursor() as $invoice) {
        // ヘッダに合計額があるので先に計算しておく
        if (!empty($invoice->quantity) && !empty($invoice->unit_price)) {
          $total_price = $total_price + round($invoice->unit_price * $invoice->quantity);
        }
      }
      $tax = $params['tax'] / 100;

      $pdf = \PDF::loadView('pdf/billing/output', ['invoices' => $invoices, 'total_price' => $total_price, 'tax' => $tax, 'billing' => $billing, 'params' => $params]);
      $pdf->setPaper('A4');

      // return $pdf->stream();
      $filename = '御請求書_'.$billing->name.'御中_'.date('Ymd', strtotime($params['from_at'])).'-'.date('Ymd', strtotime($params['to_at'])).'.pdf';
      return $pdf->download($filename);
    }
}
