<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\ReportRequest;
use App\Models\Report;
use App\Models\User;
use App\Models\Car;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Common\SendErrorMessage;

class ReportController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'start_at';
          $params['direction'] = 'desc';
        }
        if (empty($params['user_id'])) {
          $params['user_id'] = Auth::user()->id;
        }
        if (empty($params['start_at'])) {
          $params['start_at'] = date('Y-m');
        }

        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.report.note'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $reportQuery = Report::query();
        if ($params['user_id'] !== 'all') {
          $reportQuery = $reportQuery->where('user_id', '=', $params['user_id']);
        }
        $reportQuery = $reportQuery->whereYear('start_at', '=', date('Y', strtotime($params['start_at'].'-01')));
        $reportQuery = $reportQuery->whereMonth('start_at', '=', date('m', strtotime($params['start_at'].'-01')));
        if (!empty($params['search'])) {
          $reportQuery = $reportQuery->where('note', 'like', '%'.$params['search'].'%');
        }

        // DBよりReportテーブルの値を検索
        $reports = $reportQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // セレクトボックス用にUsersを取得
        $users = User::get();

        // 取得した値をビュー「report/index」に渡す
        return view('report/index')
            ->with([
                'reports' => $reports,
                'search_target' => $search_target,
                'params' => $params,
                'users' => $users,
            ]);
    }

    public function create(Request $request)
    {
        // パラメータ取得
        $params = $request->all();

        // 空の$reportを渡す
        $report = new Report();

        // セレクトボックス用にUsersを取得
        $users = User::get();

        // セレクトボックス用にCarsを取得
        $cars = Car::where('use', '=', '1')->get();

        return view('report/create', compact('report', 'users', 'cars', 'params'));
    }
    
    public function store(ReportRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report = new Report();
        $report->user_id = $request->user_id;
        $report->car_id = $request->car_id;
        $report->start_at = $request->start_at;
        $report->end_at = $request->end_at;
        $report->start_point = $request->start_point;
        $report->end_point = $request->end_point;
        $report->start_meter = $request->start_meter;
        $report->end_meter = $request->end_meter;
        $report->light_oil = $request->light_oil;
        $report->adblue = $request->adblue;
        $report->adblue_handy = $request->adblue_handy;
        $report->alcohol_check = $request->alcohol_check;
        $report->note = $request->note;
        $report->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit(Request $request, $id)
    {
        // パラメータ取得
        $params = $request->all();

        // DBよりURIパラメータと同じIDを持つReportの情報を取得
        $report = Report::findOrFail($id);

        // セレクトボックス用にUsersを取得
        $users = User::get();

        // セレクトボックス用にCarsを取得
        $cars = Car::where('use', '=', '1')->get();

        // 取得した値をビュー「report/edit」に渡す
        return view('report/edit', compact('report', 'users', 'cars', 'params'));
    }

    public function update(ReportRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report = Report::findOrFail($id);
        $report->user_id = $request->user_id;
        $report->car_id = $request->car_id;
        $report->start_at = $request->start_at;
        $report->end_at = $request->end_at;
        $report->start_point = $request->start_point;
        $report->end_point = $request->end_point;
        $report->start_meter = $request->start_meter;
        $report->end_meter = $request->end_meter;
        $report->light_oil = $request->light_oil;
        $report->adblue = $request->adblue;
        $report->adblue_handy = $request->adblue_handy;
        $report->alcohol_check = $request->alcohol_check;
        $report->note = $request->note;
        $report->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report = Report::findOrFail($id);

        // 子テーブル連動削除
        if (!empty($report->waypoints())) {
          $report->waypoints()->each(function ($waypoint) {
            $waypoint->delete();
          });
        }
        if (!empty($report->rests())) {
          $report->rests()->each(function ($rest) {
            $rest->delete();
          });
        }
        if (!empty($report->partners())) {
          $report->partners()->each(function ($partner) {
            $partner->delete();
          });
        }

        $report->delete();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function show($id)
    {
        // DBよりURIパラメータと同じIDを持つReportの情報を取得
        $report = Report::findOrFail($id);

        // セレクトボックス用にUsersを取得
        $users = User::get();

        // セレクトボックス用にCarsを取得
        $cars = Car::get();

        // 取得した値をビュー「report/edit」に渡す
        return view('report/show', compact('report', 'users', 'cars'));
    }

}
