<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\ReportPartnerRequest;
use App\Models\Report;
use App\Models\ReportPartner;
use App\Models\Partner;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Common\SendErrorMessage;

class ReportPartnerController extends Controller
{
  public function index(Request $request, $reportId)
    {
        // チェックポイント保存
        session(['checkpoint-second' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'asc';
        }

        // ページネーション設定
        // $page_max = config('const.DEFAULT_PAGE_MAX');
        $page_max = 10;

        // クエリ成形
        $report_partnerQuery = ReportPartner::query();
        $report_partnerQuery = $report_partnerQuery->where('report_id', '=', $reportId);

        // DBよりReportPartnerテーブルの値を検索
        $report_partners = $report_partnerQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // DBよりURIパラメータと同じreportIdを持つReportの情報を取得
        $report = Report::findOrFail($reportId);
            
        // 取得した値をビュー「report/partner/index」に渡す
        return view('report/partner/index')
            ->with([
                'report_partners' => $report_partners,
                'params' => $params,
                'report' => $report,
            ]);
    }

    public function create($reportId)
    {
        // 空の$report_partnerを渡す
        $report_partner = new ReportPartner();

        // DBよりURIパラメータと同じreportIdを持つReportの情報を取得
        $report = Report::findOrFail($reportId);

        // セレクトボックス用にPartnersを取得
        $partners = Partner::get();

        return view('report/partner/create', compact('report_partner', 'report', 'partners'));
    }
    
    public function store(ReportPartnerRequest $request, $reportId)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report_partner = new ReportPartner();
        $report_partner->report_id = $reportId;
        $report_partner->partner_id = $request->partner_id;
        $report_partner->quantity = $request->quantity;
        $report_partner->car_number = $request->car_number;
        $report_partner->item_name = $request->item_name;
        $report_partner->count = $request->count;
        $report_partner->take_down = $request->take_down;
        $report_partner->save();
    
        return redirect(session()->get('checkpoint-second') ?? './');
    }

    public function edit($reportId, $partnerId)
    {
        // DBよりURIパラメータと同じIDを持つReportの情報を取得
        $report_partner = ReportPartner::findOrFail($partnerId);

        // DBよりURIパラメータと同じreportIdを持つReportの情報を取得
        $report = Report::findOrFail($reportId);

        // セレクトボックス用にPartnersを取得
        $partners = Partner::get();

        // 取得した値をビュー「report/partner/edit」に渡す
        return view('report/partner/edit', compact('report_partner', 'report', 'partners'));
    }

    public function update(ReportPartnerRequest $request, $reportId, $partnerId)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report_partner = ReportPartner::findOrFail($partnerId);
        $report_partner->report_id = $reportId;
        $report_partner->partner_id = $request->partner_id;
        $report_partner->quantity = $request->quantity;
        $report_partner->car_number = $request->car_number;
        $report_partner->item_name = $request->item_name;
        $report_partner->count = $request->count;
        $report_partner->take_down = $request->take_down;
        $report_partner->save();

        return redirect(session()->get('checkpoint-second') ?? './');
    }

    public function destroy(Request $request, $reportId, $partnerId)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $report_partner = ReportPartner::findOrFail($partnerId);
        $report_partner->delete();
    
        return redirect(session()->get('checkpoint-second') ?? './');
    }

}
