<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\TypeRequest;
use App\Models\Type;
use App\Http\Controllers\Common\SendErrorMessage;

class TypeController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }
        
        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.type.name'),
          \Lang::get('db.type.note'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $typeQuery = Type::query();
        if (!empty($params['search'])) {
          $typeQuery = $typeQuery->where('name', 'like', '%'.$params['search'].'%');
          $typeQuery = $typeQuery->orWhere('note', 'like', '%'.$params['search'].'%');
        }

        // DBよりTypeテーブルの値を検索
        $types = $typeQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「type/index」に渡す
        return view('type/index')
            ->with([
                'types' => $types,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$typeを渡す
        $type = new Type();
        return view('type/create', compact('type'));
    }
    
    public function store(TypeRequest $request)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $type = new Type();
        $type->name = $request->name;
        $type->note = $request->note;
        $type->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つTypeの情報を取得
        $type = Type::findOrFail($id);

        // 取得した値をビュー「type/edit」に渡す
        return view('type/edit', compact('type'));
    }

    public function update(TypeRequest $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $type = Type::findOrFail($id);
        $type->name = $request->name;
        $type->note = $request->note;
        $type->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $type = Type::findOrFail($id);
        $type->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

}
