<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Car;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\Common\SendErrorMessage;
use App\Rules\alpha_num_check;

class UserController extends Controller
{
    public function index(Request $request)
    {
        // チェックポイント保存
        session(['checkpoint' => url()->full()]);

        // パラメータ取得
        $params = $request->all();

        // パラメータ初期値
        if (empty($params['sort'])) {
          $params['sort'] = 'id';
          $params['direction'] = 'desc';
        }
        
        // メインレイアウトの検索バーを活性化させるため検索ワードを指定
        $targets = [
          \Lang::get('db.user.account'),
          \Lang::get('db.user.name'),
        ];
        $target = implode(', ', $targets);
        $search_target = \Lang::get('system.placeholder.search_target', ['target' => $target]);

        // ページネーション設定
        $page_max = config('const.DEFAULT_PAGE_MAX');

        // クエリ成形
        $userQuery = User::query();
        if (!empty($params['search'])) {
          $userQuery = $userQuery->where('account', 'like', '%'.$params['search'].'%');
          $userQuery = $userQuery->orWhere('name', 'like', '%'.$params['search'].'%');
        }

        // DBよりUserテーブルの値を検索
        $users = $userQuery
            ->orderBy($params['sort'], $params['direction'])
            ->paginate($page_max);

        // 取得した値をビュー「user/index」に渡す
        return view('user/index')
            ->with([
                'users' => $users,
                'search_target' => $search_target,
                'params' => $params,
            ]);
    }

    public function create()
    {
        // 空の$userを渡す
        $user = new User();

        // セレクトボックス用にCarsを取得
        $cars = Car::get();

        return view('user/create', compact('user', 'cars'));
    }
    
    public function store(Request $request)
    {
        // store用追加バリデーション
        $rules = $this->rules();  // 共通ルール
        $rules['account'][] = Rule::unique('users')->whereNull('deleted_at');
        $rules['password'] = ['required', 'string', 'between:4,20', 'confirmed', new alpha_num_check];
        $rules['password_confirmation'] = 'required';
        $request->validate($rules, [], $this->attributes());

        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $user = new User();
        $user->name = $request->name;
        $user->account = $request->account;
        $user->password = Hash::make($request->password);
        $user->car_id = $request->car_id;
        $user->phone_number = $request->phone_number;
        $user->email = $request->email;
        $user->save();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function edit($id)
    {
        // DBよりURIパラメータと同じIDを持つUserの情報を取得
        $user = User::findOrFail($id);

        // セレクトボックス用にCarsを取得
        $cars = Car::get();

        // 取得した値をビュー「user/edit」に渡す
        return view('user/edit', compact('user', 'cars'));
    }

    public function update(Request $request, $id)
    {
        // update用バリデーション
        $rules = $this->rules();  // 共通ルール
        $rules['account'][] = Rule::unique('users')->ignore($id)->whereNull('deleted_at');
        if ($request->password_change) {
          // パスワードを変更する時のみ本バリデーション追加
          $rules['password'] = ['required', 'string', 'between:4,20', 'confirmed', new alpha_num_check];
          $rules['password_confirmation'] = 'required';
        }
        $request->validate($rules, [], $this->attributes());

        // エラーメッセージ
        $error_messages = [];
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $user = User::findOrFail($id);
        $user->name = $request->name;
        $user->account = $request->account;
        if ($request->password_change) {
          // パスワードを変更する時のみ更新する
          $user->password = Hash::make($request->password);
        }
        $user->car_id = $request->car_id;
        $user->phone_number = $request->phone_number;
        $user->email = $request->email;
        $user->save();

        return redirect(session()->get('checkpoint') ?? './');
    }

    public function destroy(Request $request, $id)
    {
        // エラーメッセージ
        $error_messages = [];
        if ($id == 1) {
          // テスト用にユーザーID=1の削除をロックする。formの方でも1はaccountを操作不可にしている。どのみち全ユーザー消えたら操作不能になるので本採用しても良い。
          $error_messages[] = \Lang::get('system.error-message.non-deletable');
        }
        if (!empty($error_messages)) {
          return SendErrorMessage::validationException($request, $error_messages);
        }

        $user = User::findOrFail($id);
        $user->delete();
    
        return redirect(session()->get('checkpoint') ?? './');
    }

    public function rules()
    {
        // 共通ルール
        return [
          'account' => [
            'required',
            'string',
            new alpha_num_check,
            'between:3,20',
          ],
          'name' => 'required|string|max:50',
          'car_id' => 'nullable|integer',
          'phone_number' => 'nullable|string|max:20',
          'email' => 'nullable|email|max:100',
        ];
    }

    public function attributes()
    {
        return [
          'name' => \Lang::get('db.user.name'),
          'account' => \Lang::get('db.user.account'),
          'password' => \Lang::get('db.user.password'),
          'password_confirmation' => \Lang::get('db.user.password_confirmation'),
          'car_id' => \Lang::get('db.user.car_id'),
          'phone_number' => \Lang::get('db.user.phone_number'),
          'email' => \Lang::get('db.user.email'),
        ];
    }

    
}
